#!/bin/bash
# ================================================================================
#
# LOCALDUMP :: SQLITE
# create gzipped plain text backups from each sqlite database in
# backup-dbfiles.job
#
# --------------------------------------------------------------------------------
# ah - Axel Hahn <axel.hahn@iml.unibe.ch>
# ds - Daniel Schueler <daniel.schueler@iml.unibe.ch>
#
# 2018-02-02  ah,ds  v1.0  first lines
# 2018-02-09  ah,ds  v1.1  write a .meta file after successful backup
# ================================================================================

if [ -z $BACKUP_TARGETDIR ]; then
  echo ERROR: you cannot start `basename $0` directly
  rc=$rc+1
  exit 1
fi

# --------------------------------------------------------------------------------
# CONFIG
# --------------------------------------------------------------------------------

FILEDEFS=${DIR_JOBS}/backup-dbfiles.job

# --------------------------------------------------------------------------------
# FUNCTIONS
# --------------------------------------------------------------------------------

# make sqlite3 backups of all sqlite = ... in backup-dbfiles.job
function doSqliteBackup(){

  create_targetdir

  for DATABASE_FILE in `_j_getvar ${FILEDEFS} "sqlite"`
  do
    echo --- database $DATABASE_FILE
    if [ ! -f $DATABASE_FILE ]; then
      color error
      echo ERROR: given database file does not exist: $DATABASE_FILE
      color reset
      rc=$rc+1
    else
      file $DATABASE_FILE | cut -f 2 -d ":" | fgrep -i "sqlite" >/dev/null
      if [ $? -ne 0 ]; then
        color error
        echo ERROR: given database file is not a sqlite database $DATABASE_FILE
        color reset
        rc=$rc+1
      else
        TARGET=`get_outfile ${DATABASE_FILE}`
        TARGET=${BACKUP_TARGETDIR}/`echo ${TARGET} | sed "s#/#_#g"`.sql
        META=${TARGET}.gz.meta
        echo target: $TARGET
        sqlite3 $DATABASE_FILE .dump >${TARGET}
        fetchrc
        # $myrc is last returncode - set in fetchrc
        if [ $myrc -eq 0 ]; then
          echo -n "gzip ... "
          compress_file ${TARGET}
          echo $DATABASE_FILE >${META}
        else
          echo "ERROR occured - no gzip"
        fi
        ls -l ${TARGET}*

      fi
    fi
  done
}

# restore database dump file into database
# param  string  database dump file (gzipped)
# param  string  optional: database to import; default: database is parsed from file
function restoreByFile(){
  sMyfile=$1
  sMyDb=$2

  if [ -f "${sMyDb}" ]; then
    color error
    echo ERROR: target file already exists. Remove or rename it first.
    ls -l "${sMyDb}"
    color reset
    rc=$rc+1
  else
    color cmd
    zcat "${sMyfile}" | sqlite3 "${sMyDb}"
    fetchrc
    if [ $myrc -eq 0 ]; then
      color ok
      echo OK, restore was successful
      color reset
      ls -l "${sMyDb}"
    else
      color error
      echo ERROR while restoring backup.
      color reset
    fi
  fi

}

# --------------------------------------------------------------------------------
# MAIN
# --------------------------------------------------------------------------------


# ----- check requirements
j_requireBinary "sqlite3"   1

if [ ! -f $FILEDEFS ]; then
  echo "INFO: file definitions $FILEDEFS do not exist."
  rc=$rc+1
fi


if [ $rc -ne 0 ]; then
  rc=0
  echo "SKIP: sqlite seems not to be here"
else
  if [ "$1" = "restore" ]; then
    echo
    shift 1
    restoreByFile $*

  else
    doSqliteBackup
  fi
fi

echo $0 $* [sqlite] final returncode rc=$rc

# --------------------------------------------------------------------------------
