/**
 * ======================================================================
 * 
 *        U B D
 * 
 *        Url bound to a dom id
 * 
 * ----------------------------------------------------------------------
 * 
 * This is a helper class to update the content of a dom object by a 
 * given Url after a ttl.
 * 
 * ----------------------------------------------------------------------
 * 2022-06-26  www.axel-hahn.de  first lines...
 * 2022-09-19                    do not update if domid is hidden using _iWaitIfHidden
 * ======================================================================
 */

/**
 * Url bound to a dom id
 * @return class
 */
class ubd {
    /**
     * initialize data for a dom object
     * @param  {object}  oConfig    optional config object with those subkeys:
     *                                  domid    - id of a dom object
     *                                  url      - url to an api
     *                                  header   - http request header data
     *                                  renderer - renderer function to visualize data
     *                                  ttl      - ttl in sec (TODO)
     * @returns {undefined}
     */
    constructor(oConfig) {

        this._sDomId = '';
        this._oDomObject = false;
        this._sUrl2Fetch = false; // static value or reference of a function
        this._oHeader = {};
        this._sRenderfunction = false;
        this._iTTL = false;
        this._iWaitIfHidden = 500;

        this._oTimer = false;

        this._body = '';

        oConfig = arguments ? arguments[0] : false;


        if (oConfig) {
            if (oConfig['domid']) {
                this.setDomid(oConfig['domid']);
            }
            if (oConfig['url']) {
                this.setUrl(oConfig['url']);
            }
            if (oConfig['header']) {
                this.setHeaders(oConfig['header']);
            }
            if (oConfig['renderer']) {
                this.setRenderfunction(oConfig['renderer']);
            }
            if (oConfig['ttl']) {
                this.setTtl(oConfig['ttl']);
            }
        }
    }

    // ----------------------------------------------------------------------
    // public SETTER for properties
    // ----------------------------------------------------------------------

    /**
     * set domid that will by updated
     * @param {string} sDomid if of a domobject
     */
    setDomid(sDomid) {
            if (document.getElementById(sDomid)) {
                this._sDomId = sDomid;
                this._oDomObject = document.getElementById(sDomid);
            } else {
                this._sDomId = false;
                this._oDomObject = false;
                console.error('ERROR: setDomid("' + sDomid + '") got an invalid string - this domid does not exist.');
            }
    }

    /**
     * set a rendering function that visualized data after a http request
     * @param {string|function} oFunction reference to a function ... or false to disable rendering
     */
    setRenderfunction (oFunction) {
        this._sRenderfunction = oFunction;
    }

    /**
     * Set time to live in seconds
     * @param {int} iTTL
     */
    setTtl(iTTL) {
        this._iTTL = iTTL / 1;
        this.resetTimer();
    }

    /**
     * set an url to be requested
     * @param {string|function} sUrl  static value or reference of a function
     */
    setUrl(sUrl) {
        this._sUrl2Fetch = sUrl;
    }

    /**
     * set header obejct for 2nd param in javascript  fetch() function
     * @param {object} oHeader
     */
    setHeaders(oHeader) {
        this._oHeader = oHeader;
    }

    /**
     * helper: dump current object instance to console
     */
    dumpme() {
        console.log('---------- DUMP ubd');
        console.log(this);
        console.log('---------- /DUMP ubd');
    }
    // ----------------------------------------------------------------------
    // public ACTIONS
    // ----------------------------------------------------------------------
    /**
     * show rendered html content into set domid using the render function.
     * If no rendering function was set then the response will be written
     * directly.
     * You can override both by giving a parameter (a string with html)
     * to write that one directly. It can be used to show an error message.
     *
     * TODO:
     * other output places than innerHTML by detecting the tag e.g.
     * to use input or textarea.
     *
     * @param {string} sHtml  optional: htmlcode of an error message
     */
    render(sHtml) {
        let out = sHtml ? sHtml :
            (this._sRenderfunction
                ? this._sRenderfunction(this._body)
                : this._body
            );
        this._oDomObject.innerHTML = out;
    }

    /**
     * reset timer to update the content in dom id after reaching TTL
     * used in setTtl
     */
    resetTimer() {
        clearTimeout(this._oTimer);
        // clearInterval(this._oTimer);
        if (this._iTTL) {
            var bIsVisble=this._oDomObject.offsetParent !== null;
            var iTtl=bIsVisble ? this._iTTL * 1000 : this._iWaitIfHidden;
            let self = this;
            self._oTimer = window.setTimeout(function () { self.update(); }, iTtl);
            // self._oTimer=window.setInterval(self.update, this._iTTL*1000);
        }
    }

    /**
     * make http request and call the renderer
     */
    async update() {
        var bIsVisble=this._oDomObject.offsetParent !== null;

        let self = this;
        let url = (typeof this._sUrl2Fetch == "function") ? this._sUrl2Fetch() : this._sUrl2Fetch;
        if (url == undefined) {
            console.error("SKIP update - there is no url in this object instance (anymore) :-/");
            this.dumpme();
            return 0;
        }
        if(bIsVisble){
            try {
                let response = await fetch(url, this._oHeader);
                if (response.ok) {
                    this._body = await response.json();

                    this.render();
                } else {
                    this.render('<div class="app result1">'
                        + 'ERROR ' + response.status + ': ' + response.statusText + ' - '
                        + url
                        + '</div>');
                }
            } catch (e) {
                this.render('<div class="app result1">'
                    + 'UNKNOWN: no response from '
                    + this._sUrl2Fetch
                    + '</div>');
                console.error(e);
            }
        }
        this.resetTimer();
    }

 
    }