<?php
require_once 'build.interface.php';

/**
 * build_base class that will be extended in a rollout plugin
 * see deployment/plugins/build/*
 * 
 * @author axel
 * 
 * 2024-08-23  v1.1  Axel Hahn  php8 only; added variable types; short array syntax
 */
class build_base implements iBuildplugin
{

    /**
     * path of the build directory
     * @var string
     */
    protected string $_sBuildDir = '';

    /**
     * outputfile during build
     * @var string
     */
    protected string $_sOutfile = '';

    /**
     * id of the plugin
     * @var string
     */
    protected string $_sPluginId = '';

    /**
     * array with plugin infos
     * @var array
     */
    protected array $_aPlugininfos = [];

    /**
     * language of ui; default is "en-en"
     * @var string
     */
    protected string $_sLang = "en-en";

    /**
     * array of language texts
     * @var array
     */
    protected array $_aLang = [];


    // ---------------------------------------------------------------
    // CONSTRUCTOR
    // ---------------------------------------------------------------

    /**
     * Initialize build plugin
     * @param array $aParams  hash with those possible keys
     *                  lang         string   language, i.e. 'de'
     *                  phase        string   name of phase in a project
     *                  globalcfg    array    given global config $aConfig
     *                  projectcfg   array    project config to generate config 
     *                                        for project and all phases
     */
    public function __construct(array $aParams)
    {

        // set current plugin id - taken from plugin directory name above
        $oReflection = new ReflectionClass($this);
        $this->_sPluginId = basename(dirname($oReflection->getFileName()));

        // ----- init language
        if (isset($aParams['lang'])) {
            $this->setLang($aParams['lang']);
        } else {
            $this->setLang();
        }
        if (isset($aParams['workdir'])) {
            $this->setWorkdir($aParams['workdir']);
        }
        if (isset($aParams['outfile'])) {
            $this->setOutfile($aParams['outfile']);
        }

    }

    // ---------------------------------------------------------------
    // LANGUAGE TEXTS
    // ---------------------------------------------------------------

    /**
     * Get a translated text from lang_XX.json in plugin dir;
     * If the key is missed it returns "[KEY :: LANG]"
     * 
     * @see setLang()
     * @param string $sKey  key to find in lang file
     * @return string
     */
    protected function _t(string $sKey): string
    {
        return (isset($this->_aLang[$sKey]) && $this->_aLang[$sKey])
            ? $this->_aLang[$sKey]
            : "[ $sKey :: $this->_sLang ]"
        ;
    }

    /**
     * Set language for output of formdata and other texts.
     * This method loads the language file into a hash. The output of 
     * translated texts can be done with $this->_t("your_key")
     * 
     * @see _t()
     * @param string   $sLang  language code, i.e. "de"
     * @return boolean
     */
    public function setLang(string $sLang = ''): bool
    {
        $this->_sLang = $sLang ? $sLang : $this->_sLang;

        $oReflection = new ReflectionClass($this);
        $sFile = dirname($oReflection->getFileName()) . '/lang_' . $this->_sLang . '.json';
        $this->_aLang = (file_exists($sFile)) ? json_decode(file_get_contents($sFile), 1) : $this->_aLang;
        return true;
    }
    // ---------------------------------------------------------------
    // SETTER
    // ---------------------------------------------------------------


    /**
     * Set build dir with sources
     * @param  string  $sBuildDir     full path of the build directory
     * @return bool
     */
    public function setWorkdir(string $sBuildDir): bool
    {
        $this->_sBuildDir = $sBuildDir ? $sBuildDir : $this->_sBuildDir;
        return true;
    }

    /**
     * Set outfile name
     * @param  string  $sOutFilename  filename for output (without extension)
     * @return bool
     */
    public function setOutfile(string $sOutFilename)
    {
        $this->_sOutfile = $sOutFilename ? $sOutFilename : $this->_sOutfile;
        return true;
    }

    // ---------------------------------------------------------------
    // GETTER
    // ---------------------------------------------------------------

    /**
     * Check requirements if the plugin could work
     * @return array
     */
    public function checkRequirements(): array
    {
        return [
            'echo "ERROR: The method checkRequirements() was not implemented in the build plugin [' . $this->getId() . ']"',
            'exit 1'
        ];
    }

    /**
     * Get an array with shell commands to execute
     * @return array
     */
    public function getBuildCommands(): array
    {
        return [
            'echo "ERROR: The method getBuildCommamds() was not implemented in the build plugin [' . $this->getId() . ']"',
            'exit 1'
        ];
    }

    /**
     * Get string with current ID
     * @return string
     */
    public function getId(): string
    {
        return $this->_sPluginId;
    }

    /**
     * Get string with plugin name (taken from plugin language file)
     * @return string
     */
    public function getName():string
    {
        return $this->_t('plugin_name');
    }

    /**
     * Get string with plugin description (taken from plugin language file)
     * @return string
     */
    public function getDescription(): string
    {
        return $this->_t('description');
    }

    /**
     * Get array read from info.json
     * @return array
     */
    public function getPluginInfos(): array
    {

        if ($this->_aPlugininfos) {
            return $this->_aPlugininfos;
        }

        $oReflection = new ReflectionClass($this);
        $sFile = dirname($oReflection->getFileName()) . '/info.json';
        $this->_aPlugininfos = (file_exists($sFile))
            ? json_decode(file_get_contents($sFile), 1)
            : ['error' => "unable to read info file [$sFile]."]
        ;
        return $this->_aPlugininfos;
    }

    /**
     * Get the file extension of created output file (from plugin info.json)
     * @return string
     */
    public function getExtension(): string
    {
        $aInfos = $this->getPluginInfos();
        return isset($aInfos['extension']) ? '.' . $aInfos['extension'] : '';
    }

    /**
     * Get outfile name including extension (from plugin metadata)
     * @return string
     */
    public function getOutfile(): string
    {
        return $this->_sOutfile . $this->getExtension();
    }

    /**
     * get current build dir
     * @return string
     */
    public function getBuildDir(): string
    {
        return $this->_sBuildDir;
    }

    // ----------------------------------------------------------------------
    // INTERFACE :: RENDERER
    // ----------------------------------------------------------------------

    /**
     * Render plugin box as HTML to show in plugin overview
     * @return string
     */
    public function renderPluginBox(): string
    {
        $aInfos = $this->getPluginInfos();

        return '<strong>' . $this->getName() . '</strong> (' . $this->getId() . ')<br>
                ' . $this->getDescription();
    }
}
