<?php

/**
 * Page class
 * Render output page by replacing placeholders 
 *
 * origin is from PIMPED APACHE-STATUS
 * 
 * @author Axel Hahn
 * 
 * 2024-08-23  v1.1  Axel Hahn  php8 only; added variable types; short array syntax
 */
class Page
{

    /**
     * Output type of content
     * @var string
     */
    private string $sType = 'html';

    /**
     * Array of strings for http response header
     * @var array
     */
    private array $aResponseHeader = [];

    /**
     * Replacements in the template
     * @var array
     */
    private array $aReplace = [
        '{{HEADER}}' => '',
        '{{CONTENT}}' => '',
        '{{FOOTER}}' => '',
        '{{JSONREADY}}' => '',
    ];

    /**
     * Constructor
     */
    public function __construct()
    {
        $this->setOutputtype();
    }

    /**
     * Wrap on document ready instructions in jQuery style
     * @return string
     */
    private function _finalizeJsOnReady(): string
    {
        return $this->aReplace["{{JSONREADY}}"] = '
            <script>
                $(document).ready(function() {
                    ' . $this->aReplace["{{JSONREADY}}"] . '
                } );
            </script>';
    }

    // ----------------------------------------------------------------------
    // GETTER
    // ----------------------------------------------------------------------

    /**
     * Get current page content
     * @return string HTML code
     */
    public function getContent(): string
    {
        return $this->aReplace['{{CONTENT}}'];
    }

    /**
     * Get current footer
     * @return string
     */
    public function getFooter(): string
    {
        return $this->aReplace['{{FOOTER}}'];
    }

    /**
     * Get current header in response body
     * @return string
     */
    public function getHeader(): string
    {
        return $this->aReplace['{{HEADER}}'];
    }

    /**
     * Get on ready javascript instructions
     * @return string
     */
    public function getJsOnReady(): string
    {
        return $this->aReplace['{{JSONREADY}}'];
    }


    /**
     * Get output type
     * @return string
     */
    public function getOutputtype()
    {
        return $this->sType;
    }

    // ----------------------------------------------------------------------
    // SETTER
    // ----------------------------------------------------------------------

    /**
     * Add javascript for on ready execution
     * @param string $s  javascript code
     * @return bool
     */
    public function addJsOnReady(string $s): string
    {
        $this->aReplace['{{JSONREADY}}'] .= $s;
        return true;
    }

    /**
     * add a http response header line
     * @param string $s
     * @return boolean
     */
    public function addResponseHeader(string $s)
    {
        $this->aResponseHeader[] = $s;
        return true;
    }

    /**
     * Set html body; it replaces old content
     * @param string $s  html code
     * @return boolean
     */
    public function setContent(string $s): bool
    {
        $this->aReplace['{{CONTENT}}'] = $s;
        return true;
    }
    /**
     * Set footer in html body; it replaces old content
     * @param string $s  html code
     * @return boolean
     */
    public function setFooter(string $s): bool
    {
        $this->aReplace['{{FOOTER}}'] = $s;
        return true;
    }

    /**
     * Set html header; it replaces old content
     * @param string $s  html code
     * @return boolean
     */
    public function setHeader(string $s): bool
    {
        $this->aReplace['{{HEADER}}'] = $s;
        return true;
    }

    /**
     * Set javascript code on ready; it replaces old content
     * @param string $s  javascript code
     * @return boolean
     */
    public function setJsOnReady(string $s): bool
    {
        $this->aReplace['{{JSONREADY}}'] = $s;
        return true;
    }

    /**
     * Set output type of response
     * @param string $sOutputType
     * @return boolean
     */
    public function setOutputtype(string $sOutputType = 'html'): bool
    {
        $this->sType = $sOutputType;
        return true;
    }

    // ----------------------------------------------------------------------
    // OUTPUT
    // ----------------------------------------------------------------------

    /**
     * Send http reponse headers and built the response body
     * @return string
     */
    public function render(): string
    {
        $aS = []; // search
        $aR = []; // replace

        $this->_finalizeJsOnReady();

        foreach ($this->aReplace as $sSeach => $sReplace) {
            $aS[] = $sSeach;
            $aR[] = $sReplace;
        }

        $sTemplate = false;
        $sTplFile = dirname(__FILE__) . "/" . $this->sType . ".tpl.php";
        if (!file_exists($sTplFile)) {
            throw new Exception("ERROR: template for type " . $this->sType . " was not found: $sTplFile");
        }

        $sTemplate = file_get_contents($sTplFile);
        if (!$sTemplate) {
            throw new Exception("ERROR: template file $sTplFile is empty or could not be read.");
        }

        foreach ($this->aResponseHeader as $sHeader) {
            header($sHeader);
        }
        return str_replace($aS, $aR, $sTemplate);
    }

}
