<?php

/**
 * 
 * Rollout plugin - awx
 * 
 * Run an Https POST request to AWX
 *
 * @author <axel.hahn@iml.unibe.ch>
 */
class rollout_awx extends rollout_base
{

    // url part for AWX API request to set count of results per page
    protected $_sAwxApiPaging = '&page_size=10000&page=1';

    /**
     * check requirements if the plugin could work
     */
    public function checkRequirements(): array
    {
        // no specific checks needed ... always empty
        return [];
    }

    /**
     * check access to a deploy target
     */
    public function checkConnectionToTarget()
    {
        // do nothing ... always empty
        return [];
    }

    /**
     * make an http get request and return the response body
     * it is called by _makeRequest
     * $aRequest contains subkeys
     * - url
     * - method; one of GET|POST|PUT|DELETE
     * - postdata; for POST only
     * 
     * @param array   $aRequest   arrayurl for Foreman API
     * @return array
     */
    protected function _httpRequest($aRequest = false, $iTimeout = 5)
    {

        if (!function_exists("curl_init")) {
            die("ERROR: PHP CURL module is not installed.");
        }
        $aConfig = $this->getConfig();


        $ch = curl_init($aConfig['url'] . $aRequest['url']);

        curl_setopt($ch, CURLOPT_CUSTOMREQUEST, $aRequest['method']);
        if ($aRequest['method'] === 'POST') {
            curl_setopt($ch, CURLOPT_POSTFIELDS, $aRequest['postdata']);
        }

        if ($aConfig['user']) {
            curl_setopt($ch, CURLOPT_USERPWD, $aConfig['user'] . ':' . $aConfig['password']);
        }

        if (isset($aConfig['ignore-ssl-error']) && $aConfig['ignore-ssl-error']) {
            curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 0);
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, 0);
        }

        curl_setopt($ch, CURLOPT_TIMEOUT, $iTimeout);
        curl_setopt($ch, CURLOPT_USERAGENT, 'IML Deployment :: rollout plugin awx ' . __CLASS__);
        curl_setopt($ch, CURLOPT_HEADER, 1);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);

        $res = curl_exec($ch);

        $aReturn = ['info' => curl_getinfo($ch), 'error' => curl_error($ch)];
        curl_close($ch);

        $sHeader = substr($res, 0, $aReturn['info']['header_size']);
        $aReturn['header'] = explode("\n", $sHeader);
        $aReturn['body'] = str_replace($sHeader, "", $res);

        // print_r($aReturn);
        return $aReturn;
    }

    /**
     * get AWX inventories and return them as array for select box
     * [id] => array('value' => [ID], 'label' => [NAME] [ID])
     * @return bool|array
     */
    public function getAwxInventories()
    {
        $aResponse = $this->_httpRequest(
            array(
                'url' => '/inventories/?order_by=name' . $this->_sAwxApiPaging,
                'method' => 'GET',
            )
        );

        if (!isset($aResponse['info']['http_code']) || $aResponse['info']['http_code'] !== 200) {
            return false;
        }

        $aData = json_decode($aResponse['body'], 1);
        $aReturn = [];
        if (!$aData || !isset($aData['count'])) {
            $aReturn[] = [
                'value' => false,
                'label' => '!!! Access to awx api failed !!!'
            ];
            return $aReturn;
        }
        if (count($aData['results']) < $aData['count']) {
            $aReturn[] = [
                'value' => false,
                'label' => '>>>>>>>>> WARNING: fetched ' . count($aData['results']) . ' of ' . $aData['count'] . ' items only'
            ];
        }

        foreach ($aData['results'] as $aItem) {
            $aReturn[$aItem['id']] = [
                'value' => $aItem['id'],
                'label' => $aItem['name'] . ' (id: ' . $aItem['id'] . ')'
            ];
        }

        return $aReturn;
    }
    /**
     * get AWX Job Templates and return them as array for select box
     * [id] => array('value' => [ID], 'label' => [PLAYBOOK] [ID])
     * @return bool|array
     */
    public function getAwxJobTemplates()
    {
        $aResponse = $this->_httpRequest(
            array(
                'url' => '/job_templates/?order_by=name' . $this->_sAwxApiPaging,
                'method' => 'GET',
            )
        );

        if (!isset($aResponse['info']['http_code']) || $aResponse['info']['http_code'] !== 200) {
            return false;
        }

        $aData = json_decode($aResponse['body'], 1);
        $aReturn = [];
        if (!$aData || !isset($aData['count'])) {
            $aReturn[] = [
                'value' => false,
                'label' => '!!! Access to awx api failed !!!'
            ];
            return $aReturn;
        }
        if (count($aData['results']) < $aData['count']) {
            $aReturn[] = [
                'value' => false,
                'label' => '>>>>>>>>> WARNING: fetched ' . count($aData['results']) . ' of ' . $aData['count'] . ' items only'
            ];
        }
        foreach ($aData['results'] as $aItem) {
            $aReturn[$aItem['id']] = [
                'value' => $aItem['id'],
                'label' => $aItem['name'] . ' (id: ' . $aItem['id'] . '; ' . $aItem['playbook'] . ')'
            ];
        }
        return $aReturn;
    }


    /**
     * get array with commands to execute to deploy a package
     * 
     * @param  string   $sPhase
     * @param  boolean  $bMask   Flag for public output; if true then mask your secrets
     * @return array
     */
    public function getDeployCommands($sPhase, $bMask = false)
    {
        $aReturn = array();
        $aConfig = $this->getConfig($sPhase, $bMask);

        // ----- Checks:
        $sCmdChecks = '';
        if (isset($aConfig['extravars']) && $aConfig['extravars']) {
            $aTmp = json_decode($aConfig['extravars'], 1);
            if (!$aTmp || !is_array($aTmp) || !count($aTmp)) {
                $sCmdChecks .= 'echo "ERROR: Value in extravars has wrong Syntax - this is no JSON: ' . $aConfig['extravars'] . '"; exit 1; ';
            }
            $aConfig['extravars'] = json_encode($aTmp);
        }

        if (!isset($aConfig['inventory']) || !(int) $aConfig['inventory']) {
            $sCmdChecks .= 'echo "ERROR: no awx inventory was given."; exit 1; ';
        }

        // ----- Send variables having values only
        $aBodyvars = array();
        foreach (['inventory' => 'inventory', 'limit' => 'limit', 'job_tags' => 'tags', 'extra_vars' => 'extravars'] as $sParam => $sVarkey) {
            if (isset($aConfig[$sVarkey]) && $aConfig[$sVarkey]) {
                $aBodyvars[$sParam] = $aConfig[$sVarkey];
            }
        }

        $sAuth = ($aConfig['user'] ? '--user ' . $aConfig['user'] . ':' . $aConfig['password'] : '');
        $aReturn[] = $sCmdChecks . "curl -f -k -H 'Content-Type: application/json' -XPOST -d '" . json_encode($aBodyvars, JSON_PRETTY_PRINT) . "' $sAuth " . $aConfig['url'] . "/job_templates/" . $aConfig['jobtemplate'] . "/launch/";
        return $aReturn;
    }


}
