<?php
require_once 'build.interface.php';

/**
 * build_base class that will be extended in a rollout plugin
 * see deployment/plugins/build/*
 * 
 * @author axel
 */
class build_base implements iBuildplugin{
    
    protected $_sBuildDir = false;
    protected $_sOutfile = false;

    protected $_sLang = "en";
    protected $_aLang = [];


    // ---------------------------------------------------------------
    // CONSTRUCTOR
    // ---------------------------------------------------------------

    /**
     * initialize build plugin
     * @param array $aParams  hash with those possible keys
     *                  lang         string   language, i.e. 'de'
     *                  phase        string   name of phase in a project
     *                  globalcfg    array    given global config $aConfig
     *                  projectcfg   array    project config to generate config 
     *                                        for project and all phases
     * @return boolean
     */
    public function __construct($aParams) {
        
        // set current plugin id - taken from plugin directory name above
        $oReflection=new ReflectionClass($this);
        $this->_sPluginId=basename(dirname($oReflection->getFileName()));
   
        // ----- init language
        if (isset($aParams['lang'])){
            $this->setLang($aParams['lang']);
        } else {
            $this->setLang();
        }
        if (isset($aParams['workdir'])){
            $this->setWorkdir($aParams['workdir']);
        }
        if (isset($aParams['outfile'])){
            $this->setOutfile($aParams['outfile']);
        }

        return true;
    }
    
    // ---------------------------------------------------------------
    // LANGUAGE TEXTS
    // ---------------------------------------------------------------
    
    /**
     * get a translated text from lang_XX.json in plugin dir;
     * If the key is missed it returns "[KEY :: LANG]"
     * 
     * @see setLang()
     * @param string $sKey  key to find in lang file
     * @return string
     */
    protected function _t($sKey){
        return (isset($this->_aLang[$sKey]) && $this->_aLang[$sKey])
                ? $this->_aLang[$sKey]
                : "[ $sKey :: $this->_sLang ]"
        ;
    }

    /**
     * set language for output of formdata and other texts.
     * This method loads the language file into a hash. The output of 
     * translated texts can be done with $this->_t("your_key")
     * 
     * @see _t()
     * @param string   $sLang  language code, i.e. "de"
     * @return boolean
     */
    public function setLang($sLang=false){
        $this->_sLang=$sLang ? $sLang : $this->_sLang;
        
        $oReflection=new ReflectionClass($this);
        $sFile=dirname($oReflection->getFileName()) . '/lang_'.$this->_sLang.'.json';
        $this->_aLang=(file_exists($sFile)) ? json_decode(file_get_contents($sFile), 1) : $this->_aLang;
        return true;
    }
    // ---------------------------------------------------------------
    // SETTER
    // ---------------------------------------------------------------


    /**
     * set build dir with sources
     * @param  string  $sBuildDir     full path of the build directory
     * @return array
     */
    public function setWorkdir($sBuildDir){
        return $this->_sBuildDir=$sBuildDir ? $sBuildDir : $this->_sBuildDir;
    }

    /**
     * set outfile name
     * @param  string  $sOutFilename  filename for output (without extension)
     * @return array
     */
    public function setOutfile($sOutFilename){
        return $this->_sOutfile=$sOutFilename ? $sOutFilename.$this->getExtension() : $this->_sOutfile;
    }

    // ---------------------------------------------------------------
    // GETTER
    // ---------------------------------------------------------------
    
    /**
     * check requirements if the plugin could work
     * @return array
     */
    public function checkRequirements() {
        return [
            'echo "ERROR: The method checkRequirements() was not implemented in the build plugin ['.$this->getId().']"',
            'exit 1'
            ];
    }

    /**
     * get an array with shell commands to execute
     * @return array
     */
    public function getBuildCommands(){
        return [
            'echo "ERROR: The method getBuildCommamds() was not implemented in the build plugin ['.$this->getId().']"',
            'exit 1'
            ];
    }
    
    /**
     * get string with current ID
     * @return string
     */
    public function getId(){
        return $this->_sPluginId;
    }
    
    /**
     * get string with plugin name (taken from plugin language file)
     * @return string
     */
    public function getName(){
        return $this->_t('plugin_name');
    }
    
    /**
     * get string with plugin description (taken from plugin language file)
     * @return string
     */
    public function getDescription(){
        return $this->_t('description');
    }
    /**
     * get array read from info.json
     * @return type
     */
    public function getPluginInfos(){

        if ($this->_aPlugininfos){
            return $this->_aPlugininfos;
        }
        
        $oReflection=new ReflectionClass($this);
        $sFile=dirname($oReflection->getFileName()) . '/info.json';
        $this->_aPlugininfos= (file_exists($sFile))
            ? json_decode(file_get_contents($sFile), 1)
            : array('error'=> 'unable to read info file ['.$sFile.'].')
        ;
        return $this->_aPlugininfos;
    }

    /**
     * get the file extension of created output file (from plugin info.json)
     */
    public function getExtension(){
        $aInfos=$this->getPluginInfos();
        return isset($aInfos['extenmsion']) ? '.'.$aInfos['extenmsion'] : '';
    }
    /**
     * set outfile name
     * @param  string  $sOutFilename  filename for output (without extension)
     * @return array
     */
    public function getOutfile(){
        return $this->_sOutfile;
    }
    /**
     * set outfile name
     * @param  string  $sOutFilename  filename for output (without extension)
     * @return array
     */
    public function getBuildDir(){
        return $this->_sBuildDir;
    }

    // ----------------------------------------------------------------------
    // INTERFACE :: RENDERER
    // ----------------------------------------------------------------------
    public function renderPluginBox(){
        $sReturn='';        
        $aInfos=$this->getPluginInfos();

        return '<strong>'.$this->getName().'</strong> ('.$this->getId().')<br>
                '.$this->getDescription();
    }
}
