<?php
/**
 * ----------------------------------------------------------------------
 *   _____ __  __ _        _____          _ _               _   
 *  |_   _|  \/  | |      |  __ \        | (_)             | |  
 *    | | | \  / | |      | |__) |___  __| |_ _ __ ___  ___| |_ 
 *    | | | |\/| | |      |  _  // _ \/ _` | | '__/ _ \/ __| __|
 *   _| |_| |  | | |____  | | \ \  __/ (_| | | | |  __/ (__| |_ 
 *  |_____|_|  |_|______| |_|  \_\___|\__,_|_|_|  \___|\___|\__|
 *
 * ----------------------------------------------------------------------
 * Loads a config json from outside webroot and makes a 3xx redirect
 * if the definition exists
 * ----------------------------------------------------------------------
 * errorcodes:
 * - code 500 - no config or statuscode is not supported
 * - code 404 - the redirect does not exist in the json file or is 
 *              incomplete (no target, no http status code)
 * ----------------------------------------------------------------------
 * ah=<axel.hahn@iml.unibe.ch>
 * 2019-04-23  v1.0  ah  first version (without tracking)
 * 2019-04-25  v1.1  ah  added regex handling; added GET param "debugredirect"
 * 2019-04-25  v1.2  ah  use REQUEST_URI (works on Win and Linux)
 * 2020-05-06  v1.3  ah  added aliases for multiple domains with the same config
 * 2020-05-06  v1.4  ah  rewrite as class
 * 2023-08-28  v1.5  ah  fix loop over config with missing regex section.
 */

/**
 * Description of redirect
 *
 * @author axel
 */
class redirect {

    // ----------------------------------------------------------------------
    // CONFIG
    // ----------------------------------------------------------------------

    protected $bDebug = false;
    protected $sConfigDir = __DIR__ . '/../../config';
    protected $sAbout = 'IML redirect <small>v1.4</small>';

    protected $sHostname = false;
    protected $sRequest = false;

    protected $sCfgfile = false;
    protected $aConfig  = false;
    protected $aRedirect  = false;
    
    public $urlRepo='https://git-repo.iml.unibe.ch/iml-open-source/redirect-handler';
    public $urlDocs='https://os-docs.iml.unibe.ch/redirect-handler/';

    // ----------------------------------------------------------------------
    // CONSTRUCTOR
    // ----------------------------------------------------------------------
    public function __constructor($sHostname=false, $sRequest=false) {
        if($sHostname){
            $this->setHost($sHostname);
        }
        if($sRequest){
            $this->setRequest($sRequest);
        }
        
        return true;
    }

    
    // ----------------------------------------------------------------------
    // DEBUG
    // ----------------------------------------------------------------------
    
    /**
     * write a debug message into the header
     * @global    boolean $bDebug         flag: show debug infos?
     * @staticvar int     $i              counter
     * @param     string  $sDebugMessage  message to show
     * @return boolean
     */
    protected function _wd($sDebugMessage) {
        
        if (!$this->bDebug) {
            return false;
        }
        static $i;
        $i++;
        header('X-DEBUG-' . $i . ': ' . $sDebugMessage);
        return true;
    }
    // ----------------------------------------------------------------------
    // SET INTERNAL VARS
    // ----------------------------------------------------------------------
    /**
     * get a string with full path of a config file with aliases
     * @return string
     */
    protected function _generateAliasfile() {
        return $this->sConfigDir . '/aliases.json';
    }
    /**
     * get a string with full path of a config file based on hostname
     * @param string $sHostname
     * @return string
     */
    protected function _generateCfgfile($sHostname) {
        
        return $this->sConfigDir . '/redirects_' . $sHostname . '.json';
    }
    
    protected function _getAliases(){
        $sAliasfile=$this->_generateAliasfile();
        $this->_wd('check alias file '.$sAliasfile);
        if(!file_exists($sAliasfile)){
            $this->_wd('alias do not exist');
            // $this->sendBody(500, '<h1>Internal Server Error</h1>Errorcode 01');
            return false;
        }
        $aAliases = json_decode(file_get_contents($sAliasfile), 1);
        return $aAliases;
    }

    /**
     * get an array with redirect config based on a given hostname
     * @param string  $sHostname  hostname
     * @param boolean $bAbort     flag: true = do not scan aliases (used for loop detection)
     * @return array
     */
    protected function _getConfig() {
        $this->aConfig=false;
        $this->aRedirect=false;
        $this->_getEffectiveConfigfile();
        if($this->sCfgfile){
            $aConfig=json_decode(file_get_contents($this->sCfgfile), 1);
            if (!is_array($aConfig) || !count($aConfig)) {
                $this->_wd('no config available');
                // $this->sendBody(500, '<h1>Internal Server Error</h1>Errorcode 02');
            }
            $this->aConfig=$aConfig;
        }
        return true;
    }
    
    /**
     * get an array with redirect config based on a given hostname
     * or detection in the alias config
     * @param string  $sHostname  hostname
     * @param boolean $bAbort     flag: true = do not scan aliases (used for loop detection)
     * @return array
     */
    protected function _getEffectiveConfigfile($sHostname=false, $bAbort=false) {
        if(!$sHostname){
            $sHostname=$this->sHostname;
        }
        $this->sCfgfile=false;
        $sCfgfile = $this->_generateCfgfile($sHostname);
        $this->_wd('check config '.$sCfgfile);
        if (!file_exists($sCfgfile)) {
            if($bAbort){
                // $this->sendBody(500, '<h1>Internal Server Error</h1>Errorcode 01');
                return false;
            }
            $aAliases = $this->_getAliases();
            $this->_wd('checking aliases '.print_r($aAliases,1));
            if (!isset($aAliases[$sHostname]) || !file_exists($this->_generateCfgfile($aAliases[$sHostname]))){
                $this->_wd('sorry no valid alias for '. $sHostname);
                // $this->sendBody(500, '<h1>Internal Server Error</h1>Errorcode 01');
                return false;
            }
            // remark: with abort flag
            return $this->_getEffectiveConfigfile($aAliases[$this->sHostname], 1);

        }
        $this->sCfgfile=$sCfgfile;
        return $sCfgfile;
    }
    
    
    /**
     * enable/ disable debug
     * @param boolean $bEnable
     */
    public function setDebug($bEnable){
        $this->bDebug=!!$bEnable;
        return true;
    }
    
    /**
     * set hostname; internally it detects the config too
     * @param type $sHostname
     * @return boolean
     */
    public function setHost($sHostname){
        $this->sHostname=$sHostname;
        $this->_getConfig();
        return true;
    }
    
    /**
     * set the request
     * @param type $sRequest
     * @return boolean
     */
    public function setRequest($sRequest){
        $this->sRequest=$sRequest;
        $this->aRedirect=false;
        return true;
    }

    // ----------------------------------------------------------------------
    // FUNCTIONS - GET REDIRECT
    // ----------------------------------------------------------------------

    /**
     * get an array with the matching redirect; it returns false if none was 
     * detected
     * @return array
     */
    public function getRedirect(){
        if(is_array($this->aRedirect)){
            return $this->aRedirect;
        }
        
        $aRedirect = false;
        
        // remark:
        // $this->aConfig is set in setHost() with $this->_getConfig();
        
        if (isset($this->aConfig['direct'][$this->sRequest])) {
            $this->_wd("DIRECT MATCH");
            $aRedirect = $this->aConfig['direct'][$this->sRequest];
        } else {
            $this->_wd("no direct match ... scanning regex");
            if(isset($this->aConfig['regex']) && is_array($this->aConfig['regex'])){
                foreach (array_keys($this->aConfig['regex']) as $sRegex) {
                    $this->_wd("check if regex [$sRegex] matches $this->sRequest");
                    if (preg_match('#' . $sRegex . '#', $this->sRequest)) {
                        $this->_wd("REGEX MATCH! aborting tests");
                        $aRedirect = $this->aConfig['regex'][$sRegex];
                        break;
                    }
                }
            }
        }
        $this->aRedirect=$aRedirect;
        return $aRedirect;
    }
    
    public function getRedirectCode($iNone=false){
        $aRedirect=$this->getRedirect();
        return isset($aRedirect['code']) ? $aRedirect['code'] : $iNone;
    }
    public function getRedirectTarget(){
        $aRedirect=$this->getRedirect();
        return isset($aRedirect['target']) ? $aRedirect['target'] : false;
    }
    

    
    // ----------------------------------------------------------------------
    // FUNCTIONS - SEND DATA
    // ----------------------------------------------------------------------

    /**
     * make the redirect if it was found ... or a 404
     * @return true
     */
    public function makeRedirect(){
        $sTarget=$this->getRedirectTarget();
        if(!$sTarget){
            $this->_wd('send a not found');
            $this->sendBody(404, '<h1>404 Not found</h1>');
        } else {
            $iCode=$this->getRedirectCode();
            $this->_wd("Redirect with http status [" . $iCode . "] to new Location: " . $sTarget);
            $this->sendHttpStatusheader($iCode);
            header("Location: " . $sTarget);
        }
        return true;
    }
    
    /**
     * send http status header
     * @param integer $iCode http code
     * @return boolean
     */
    public function sendHttpStatusheader($iCode) {
        $aHeaders = array(
            301 => 'Moved Permanently',
            302 => 'Found', // (Moved Temporarily) 
            303 => 'See other', // redirect with GET
            307 => 'Temporary Redirect',
            308 => 'Permanent Redirect',
            404 => 'Not found',
            410 => 'Gone',
            500 => 'Internal Server Error',
        );

        $iCode = (int) $iCode;
        if (!isset($aHeaders[$iCode])) {
            $this->sendBody(500, '<h1>Internal Server Error</h1>Errorcode 03: ' . $iCode);
        }
        header("HTTP/1.0 $iCode " . $aHeaders[$iCode]);
        return true;
    }

    /**
     * send http header for given status code, show document and exit
     * @see sendHttpStatusheader()
     * 
     * @param integer  $iCode  http status code
     * @param stringg  $sBody  message text as html code
     */
    public function sendBody($iCode, $sBody) {
        $this->sendHttpStatusheader($iCode);
        die('<!doctype html><html><head>'
                . '<title>Redirect</title>'
                . '<style>'
                . 'body{background:#eee; background: linear-gradient(-10deg,#ccc,#eee,#ddd) fixed; color:#444; font-family: verdana,arial;}'
                . 'h1{color:#a44;font-size: 300%;border-bottom: 1px solid #fff;}'
                . 'h2{color:#ccc; color: rgba(0,0,0,0.1); font-size: 300%; position: absolute; right: 1em; bottom: 1em; text-align: right;}'
                . 'h2 small{font-size: 50%;}'
                . 'footer{background:#ccc; bottom: 1em; color:#666; position: absolute; padding: 1em; right: 1em; }'
                . '</style>'
                . '</head>'
                . '<body>'
                . $sBody
                . '<h2>' . $this->sAbout . '</h2>'
                . '<footer>&copy; ' . date('Y') . ' '.$this->sHostname.'</footer>'
                . '</body></html>'
        );
    }


}

